/* Simpler scanner for EPSP  */

/* 

Written by: Daniel M. German <dmg@csg.uwaterloo.ca>
Date: May 26, 1999 

This scanner is intended to make life easier for the students of
the Databases Course. 

Get_Token reads a "token" from the command line.  

There are 5 types of tokens: ALPHA, NUMERIC, OTHER, END_OF_LINE, END_OF_FILE.

Alpha are alphanumeric as decided by the isalpha function. Numeric are
sequences of digits which might be proceeded by + or - signs.
OTHER are sequences of any other characters.

Spaces and tabs are considered separators and are not reported back to the 
reader.

END_OF_LINE and END_OF_FILE are self explanatory.

The result token is put "token" parameter. Get_Token assumes that
there is enough space in this string. The calling function should
make sure there is enough space in "token". 


*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <stdlib.h>
#include "epspscan.h"




#define TRUE 1
#define FALSE 0

static int IsLastCharEO = FALSE;
static int Is_Empty_Space(int c)
{
  switch (c) {
  case '\r':
  case ' ':
  case '\t':
    return TRUE;
  default:
    return FALSE;
  }
}


/* 
This functions reads everything until the end of line, it is useful
in case there is an error (input records are separated by line endings)
*/
void Reset_Get_Token(void)
{
  unsigned int c;
  if (IsLastCharEO) {
    /* If it was already a end of line or end of file, then do nothing */
    return;
  }
  while ( (c= getchar()) != '\n' && c != EOF)
    ;
  IsLastCharEO = 1; /* To make sure we are not called again */
}

/* 
   This function reads a token from the standard input. 
   It returns the type of the token (as defined in token_type) 
   and sets "token" parameter to the input string. It assumes
   that token has been allocated and there is enough space
   for the input token. IT DOES NOT ALLOCATE IT.
 */
token_type Get_Token(char * token)
{
    int iIndex = 0;
    int c;
    int otherc;
    /* Read next character in input stream */
    IsLastCharEO = 0;    
    while (Is_Empty_Space(c=getchar())) {
      /* Consume empty spaces */
      
    }
    /* Check if there is something to read */
    switch (c) {
    case EOF:
      IsLastCharEO = 1;
      token[iIndex] =0 ; 
      return END_OF_FILE;
    case '\n':
      IsLastCharEO = 1;
      token[iIndex] = 0;
      return END_OF_LINE;
    }

    /*now, we have to check for the + and - sign */
    if (c == '+' || c == '-') {
      /* We can't decide. We have to read another char */
      otherc = getchar();
      if (isdigit(otherc))      {
        /* It is a number
           Save current char in the buffer
           and keep reading */
        token[iIndex++] = (char)c;
        c = otherc;
      }
      else 
        /* we have to undo the read, so things keep as
           usual, +/- are part of "OTHER" */
        ungetc(otherc,stdin);
    }

    /* Different options */
    if(isalpha( c )) {
        /* Read while there are alphas and digits */
        do {
          token[iIndex++] = (char)c;
        } while (isalpha(c=getchar()) || isdigit(c));
        /* The last one is not part of the token,
           return it! */
        ungetc(c,stdin);
        token[iIndex] = 0;
        return ALPHA;
    }
    else if( isdigit(c)) {
        /* Read while there are digits */
        do {
          token[iIndex++] = (char)c;
        } while (isdigit(c=getchar()));
        /* The last one is not part of the token,
           return it! */
        ungetc(c,stdin);
        
        
        token[iIndex] = 0;
        return NUMERIC;
    }
    else {
        /* Read everything that looks strange */
        do {
          token[iIndex++] = (char)c;
        } while (!(Is_Empty_Space(c=getchar()) ||
                   isalpha(c) || isdigit(c) || c == '\n'));
        /* Return last character */
        ungetc(c,stdin);
        token[iIndex] = 0;
        return OTHER;

    }
    /* It should never reach this line */
}





